import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import { requireAdmin } from "@/lib/admin";
import { z } from "zod";
import * as bcrypt from "bcryptjs";
import crypto from "crypto";
import { derivePseudoAddress } from "@/lib/pseudoAddress";
import { generatePasswordResetToken, getPasswordResetExpiration } from "@/lib/passwordReset";
import { sendWelcomeEmail } from "@/lib/email";

const Body = z.object({
  email: z.string().email(),
  name: z.string().min(1).max(100).optional(),
  password: z.string().min(6).max(128).optional(), // Optional - will generate temporary password if not provided
  role: z.enum(["TECHNICIAN", "USER"]).default("TECHNICIAN"),
  ethAddress: z
    .string()
    .regex(/^0x[a-fA-F0-9]{40}$/, "Invalid Ethereum address")
    .optional(),
  hourlyRate: z.preprocess(
    (val) => (val === "" || val === null || val === undefined ? null : val),
    z.union([z.number().min(0).max(1000), z.null()]).optional()
  ),
  customRoleId: z.preprocess(
    (val) => (val === "" || val === null || val === undefined ? null : val),
    z.union([z.string(), z.null()]).optional()
  ),
  isExternal: z.boolean().optional().default(false),
  companyName: z.string().max(200).nullable().optional(),
});

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await requireAdmin(req, res);
  if (!session) return;
  if (req.method !== "POST")
    return res.status(405).json({ ok: false, error: "Method not allowed" });

  const parsed = Body.safeParse(req.body);
  if (!parsed.success)
    return res.status(400).json({
      ok: false,
      error: "Invalid input",
      details: parsed.error.flatten(),
    });

  const { email, name, password, role, ethAddress, hourlyRate, customRoleId, isExternal, companyName } =
    parsed.data;

  const exists = await prisma.user.findUnique({ where: { email } });
  if (exists)
    return res.status(409).json({ ok: false, error: "Email already in use" });

  // Generate a secure random password if not provided (user will set their own via welcome email)
  // This temporary password will never be used since we send a password reset link
  const tempPassword = password || crypto.randomBytes(32).toString("hex");
  const passwordHash = await bcrypt.hash(tempPassword, 10);

  // Convert empty string to null for customRoleId
  const finalCustomRoleId =
    customRoleId && customRoleId.trim() !== "" ? customRoleId : null;

  // Handle hourlyRate - convert to null if undefined or NaN
  const finalHourlyRate =
    hourlyRate != null && !isNaN(hourlyRate) ? hourlyRate : null;

  // Generate password reset token for welcome email (48 hour expiration)
  const resetToken = generatePasswordResetToken();
  const resetExpires = getPasswordResetExpiration(48);

  // Create first (to get a stable user.id)
  const created = await prisma.user.create({
    data: {
      email,
      name: name ?? null,
      passwordHash,
      role,
      ethAddress: null,
      hourlyRate: finalHourlyRate,
      customRoleId: finalCustomRoleId,
      isExternal: isExternal ?? false,
      companyName: companyName ?? null,
      passwordResetToken: resetToken,
      passwordResetExpires: resetExpires,
    },
    select: {
      id: true,
      email: true,
      name: true,
      role: true,
      ethAddress: true,
      hourlyRate: true,
      customRoleId: true,
      isExternal: true,
      companyName: true,
      createdAt: true,
    },
  });

  // If admin didn't provide ethAddress, derive one deterministically from the user.id
  const finalEth = ethAddress ?? derivePseudoAddress(created.id);

  const updated = await prisma.user.update({
    where: { id: created.id },
    data: { ethAddress: finalEth },
    select: {
      id: true,
      email: true,
      name: true,
      role: true,
      ethAddress: true,
      hourlyRate: true,
      customRoleId: true,
      isExternal: true,
      companyName: true,
      createdAt: true,
    },
  });

  // Send welcome email with password reset link (don't block on email failure)
  try {
    await sendWelcomeEmail(updated.email, updated.name, resetToken);
  } catch (emailError) {
    console.error("Failed to send welcome email:", emailError);
    // Don't fail the user creation if email fails
  }

  res.status(201).json({ ok: true, user: updated });
}
